<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreInvoiceRequest;
use App\Models\Doctor;
use App\Models\DoctorNcfAuthorization;
use App\Models\Insurer;
use App\Models\Invoice;
use App\Models\NcfType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class InvoiceController extends Controller
{
    public function index(Request $request)
    {
        // Para el formulario (selects)
        $doctors = Doctor::orderBy('full_name')->get(['id', 'full_name']);
        $insurers = Insurer::orderBy('name')->get(['id', 'name']);
        $ncfTypes = NcfType::where('active', true)->orderBy('id')->get(['id', 'name', 'prefix']);

        // Si quieres listar facturas debajo (opcional)
        $invoices = Invoice::with(['doctor', 'insurer', 'ncfType'])
            ->orderByDesc('id')
            ->paginate(10)
            ->withQueryString();

        return view('invoices.index', compact('doctors', 'insurers', 'ncfTypes', 'invoices'));
    }

    public function store(StoreInvoiceRequest $request)
    {
        $data = $request->validated();

        DB::transaction(function () use ($data) {

            // 1) Buscar autorización NCF del doctor para ese tipo
            $auth = \App\Models\DoctorNcfAuthorization::where('doctor_id', $data['doctor_id'])
                ->where('ncf_type_id', $data['ncf_type_id'])
                ->where('active', true)
                ->lockForUpdate()
                ->first();

            if (!$auth) {
                abort(422, 'El médico no tiene autorización NCF activa para este tipo.');
            }

            if ($auth->expires_at && now()->toDateString() > $auth->expires_at->toDateString()) {
                abort(422, 'La autorización NCF está vencida.');
            }

            $next = (int) ($auth->current_number ?? $auth->from_number);

            if ($next < (int)$auth->from_number || $next > (int)$auth->to_number) {
                abort(422, 'No hay NCF disponibles en el rango autorizado.');
            }

            // 2) Construir NCF completo: prefix + correlativo padded (8 dígitos)
            $type = \App\Models\NcfType::findOrFail($data['ncf_type_id']);
            $ncfNumber = $type->prefix . str_pad((string)$next, 8, '0', STR_PAD_LEFT);

            // 3) Total
            $total = collect($data['items'])->sum(fn($it) => (float)$it['amount']);

            // 4) Crear invoice
            $invoice = \App\Models\Invoice::create([
                'doctor_id'    => $data['doctor_id'],
                'insurer_id'   => $data['insurer_id'],
                'ncf_type_id'  => $data['ncf_type_id'],
                'invoice_date' => $data['invoice_date'],

                'ncf_seq'      => $next,
                'ncf_number'   => $ncfNumber,

                'total_amount' => $total,
                'status'       => 'issued',
                'created_by'   => auth()->id(),
            ]);

            // 5) Items
            $rows = collect($data['items'])->map(function ($it) use ($invoice) {
                return [
                    'invoice_id'       => $invoice->id,
                    'service_date'     => $it['service_date'],
                    'patient_name'     => $it['patient_name'],
                    'affiliate_no'     => $it['affiliate_no'],
                    'authorization_no' => $it['authorization_no'],
                    'procedure_id'     => $it['procedure_id'] ?? null,
                    'amount'           => $it['amount'],
                    'created_at'       => now(),
                    'updated_at'       => now(),
                ];
            })->all();

            $invoice->items()->insert($rows);

            // 6) Incrementar correlativo
            $auth->current_number = $next + 1;
            $auth->save();
        });

        return redirect()->route('invoices.index')->with('success', 'Factura creada correctamente.');
    }


    public function destroy(Invoice $invoice)
    {
        DB::transaction(function () use ($invoice) {
            $invoice->items()->delete();
            $invoice->delete();
        });

        return redirect()->route('invoices.index')->with('success', 'Factura eliminada.');
    }

    public function previewNcf(Request $request)
    {
        $doctorId = (int) $request->query('doctor_id');
        $ncfTypeId = (int) $request->query('ncf_type_id');

        if (!$doctorId || !$ncfTypeId) {
            return response()->json(['ok' => false, 'message' => 'doctor_id y ncf_type_id son requeridos.'], 422);
        }

        $auth = DoctorNcfAuthorization::where('doctor_id', $doctorId)
            ->where('ncf_type_id', $ncfTypeId)
            ->where('active', true)
            ->first();

        if (!$auth) {
            return response()->json(['ok' => false, 'message' => 'El médico no tiene autorización NCF activa.'], 404);
        }

        if ($auth->expires_at && now()->toDateString() > $auth->expires_at->toDateString()) {
            return response()->json(['ok' => false, 'message' => 'La autorización NCF está vencida.'], 422);
        }

        $next = (int) ($auth->current_number ?? $auth->from_number);

        if ($next < (int)$auth->from_number || $next > (int)$auth->to_number) {
            return response()->json(['ok' => false, 'message' => 'No hay NCF disponibles en el rango.'], 422);
        }

        $type = NcfType::findOrFail($ncfTypeId);
        $ncfNumber = $type->prefix . str_pad((string)$next, 8, '0', STR_PAD_LEFT);

        return response()->json([
            'ok' => true,
            'ncf_seq' => $next,
            'ncf_number' => $ncfNumber,
            'expires_at' => optional($auth->expires_at)->toDateString(),
        ]);
    }

    public function doctorData(Request $request)
    {
        $doctorId = (int) $request->query('doctor_id');
        if (!$doctorId) {
            return response()->json(['ok' => false, 'message' => 'doctor_id es requerido.'], 422);
        }

        $doctor = Doctor::with(['insurers' => function ($q) {
            // solo las que están vinculadas y con código (si quieres exigir código)
            $q->orderBy('name');
        }, 'ncfAuthorizations.ncfType'])->findOrFail($doctorId);

        $insurers = $doctor->insurers->map(fn($i) => [
            'id' => $i->id,
            'name' => $i->name,
            'doctor_code' => $i->pivot->doctor_code,
        ])->values();

        $ncfTypes = $doctor->ncfAuthorizations
            ->where('active', true)
            ->filter(fn($a) => !$a->expires_at || now()->toDateString() <= $a->expires_at->toDateString())
            ->map(fn($a) => [
                'id' => $a->ncfType->id,
                'name' => $a->ncfType->name,
                'prefix' => $a->ncfType->prefix,
            ])
            ->unique('id')
            ->values();

        return response()->json([
            'ok' => true,
            'insurers' => $insurers,
            'ncf_types' => $ncfTypes,
        ]);
    }

    public function show(Invoice $invoice)
{
    $invoice->load([
        'doctor',
        'insurer',
        'ncfType',
        'items',
        'createdBy',
        'payments' => function ($q) {
            $q->orderByDesc('payment_date')->orderByDesc('id');
        },
    ]);

    return response()->json([
        'ok' => true,
        'invoice' => [
            'id' => $invoice->id,
            'ncf_number' => $invoice->ncf_number,
            'invoice_date' => optional($invoice->invoice_date)->toDateString(),

            'doctor' => $invoice->doctor?->full_name,
            'insurer' => $invoice->insurer?->name,
            'ncf_type' => $invoice->ncfType?->name,

            // 👇 usa estos nombres para que tu modal no se rompa
            'total_amount' => (string) $invoice->total_amount,
            'paid_amount'  => (string) ($invoice->paid_amount ?? 0),
            'status' => $invoice->status,
            'payment_status' => $invoice->payment_status ?? 'unpaid',
            'paid_at' => optional($invoice->paid_at)->toDateTimeString(),

            'created_by' => $invoice->createdBy?->name,

            'items' => $invoice->items->map(fn($it) => [
                'service_date' => optional($it->service_date)->toDateString(),
                'patient_name' => $it->patient_name,
                'affiliate_no' => $it->affiliate_no,
                'authorization_no' => $it->authorization_no,
                'amount' => (string)$it->amount,
            ])->values(),

            // 👇 pagos para el modal
            'payments' => $invoice->payments->map(fn($p) => [
                'id' => $p->id,
                'payment_date' => optional($p->payment_date)->toDateString(),
                'amount' => (string)$p->amount,
                'method' => $p->method,
                'reference_no' => $p->reference_no,
                'notes' => $p->notes,
            ])->values(),
        ],
    ]);
}


}
